#pragma once
#include <glm/glm.hpp>
#include <vector>
#include <functional>

struct alignas(16) MaterialData {
    glm::vec3 baseColor = glm::vec3(1.0f);
    int baseColorTexture = -1;  // -1 表示无纹理

    glm::vec3 normal = glm::vec3(0.0f, 0.0f, 1.0f);
    int normalTexture = -1;

    float metallic = 0.0f;
    int metallicTexture = -1;

    float roughness = 0.5f;
    int roughnessTexture = -1;
};

inline bool operator==(const MaterialData& a, const MaterialData& b) {
    return a.baseColor == b.baseColor &&
           a.roughness == b.roughness &&
           a.normal == b.normal &&
           a.metallic == b.metallic &&
           a.baseColorTexture == b.baseColorTexture &&
           a.normalTexture == b.normalTexture &&
           a.metallicTexture == b.metallicTexture &&
           a.roughnessTexture == b.roughnessTexture;
}

// 自定义哈希函数
namespace std {
    template <>
    struct hash<MaterialData> {
        std::size_t operator()(const MaterialData& mat) const {
            size_t h = 0;
            auto hash_combine = [&](auto val) {
                h ^= std::hash<decltype(val)>{}(val) + 0x9e3779b9 + (h << 6) + (h >> 2);
            };
            hash_combine(mat.baseColor.x);
            hash_combine(mat.baseColor.y);
            hash_combine(mat.baseColor.z);
            hash_combine(mat.roughness);
            hash_combine(mat.normal.x);
            hash_combine(mat.normal.y);
            hash_combine(mat.normal.z);
            hash_combine(mat.metallic);
            hash_combine(mat.baseColorTexture);
            hash_combine(mat.normalTexture);
            hash_combine(mat.metallicTexture);
            hash_combine(mat.roughnessTexture);
            return h;
        }
    };
}

// 管理单个MaterialData，先不支持图像纹理功能，创建时拥有唯一id。不提供外部创建方法，由MaterialManager创建
class Material {
    friend class MaterialManager;

public:
    uint32_t getID() const { return m_id; }
    const MaterialData& getData() const { return m_data; }
    MaterialData& getMutableData() { return m_data; }

private:
    Material(uint32_t id, const MaterialData& data) : m_id(id), m_data(data) {}

    uint32_t m_id;
    MaterialData m_data;
};


// 全局的材质管理，单例类，提供单个材质的创建，通过id索引并返回单个材质
class MaterialManager {
public:
    static MaterialManager& get() {
        static MaterialManager instance;
        return instance;
    }

    // 创建一个新材质，返回其ID
    uint32_t createMaterial(const MaterialData& data = MaterialData()) {
        auto it = m_dataToID.find(data);
        if (it != m_dataToID.end())
            return it->second;

        uint32_t id = static_cast<uint32_t>(m_materials.size());
        m_materials.push_back(Material(id, data));
        m_dataToID[data] = id;
        return id;
    }

    // 获取材质引用（只读）
    const Material& getMaterial(uint32_t id) const {
        assert(id < m_materials.size());
        return m_materials[id];
    }

    // 获取材质引用（可修改）
    Material& getMutableMaterial(uint32_t id) {
        assert(id < m_materials.size());
        return m_materials[id];
    }

    // 返回所需SSBO大小
    uint64_t getMaterialDataSize() {
        return m_materials.size() * sizeof(MaterialData);
    }

    // 将材质数据打包为GPU上传格式
    std::vector<MaterialData> getMaterialDataBuffer() const {
        std::vector<MaterialData> buffer;
        buffer.reserve(m_materials.size());
        for (const auto& mat : m_materials)
            buffer.push_back(mat.getData());
        return buffer;
    }

private:
    std::vector<Material> m_materials;
    std::unordered_map<MaterialData, uint32_t> m_dataToID;  // 去重索引

    // 禁止外部创建
    MaterialManager() = default;
};
