#include "Scene.h"
#include "GlobalSettings.h"

void Scene::initScene()
{
    // 手动创建材质
    MaterialData mat{};
    mat.baseColor = hexToVec3("#df4c68");
    mat.roughness = 0.3f;
    mat.metallic = 0.0f;
    uint32_t material_0 = MaterialManager::get().createMaterial(mat);

    mat.baseColor = hexToVec3("#4cdfd0");
    mat.roughness = 0.3f;
    mat.metallic = 0.5f;
    uint32_t material_1 = MaterialManager::get().createMaterial(mat);

    mat.baseColor = glm::vec3(0.3f, 0.3f, 0.3f);
    mat.roughness = 0.5f;
    mat.metallic = 0.0f;
    uint32_t material_2 = MaterialManager::get().createMaterial(mat);

    // 手动创建场景物体
    auto cube = std::make_unique<Mesh>();
    cube->set_mesh_cube();
    cube->set_position(glm::vec3(1.5f, 0.0f, 0.0f));
    cube->set_materialID(material_0);
    s_meshes.push_back(std::move(cube));

    auto sphere = std::make_unique<Mesh>();
    sphere->set_mesh_sphere(32, 18, 1.0f);
    sphere->set_position(glm::vec3(-1.5f, 0.0f, 0.0f));
    sphere->set_materialID(material_1);
    s_meshes.push_back(std::move(sphere));

    auto plane = std::make_unique<Mesh>();
    plane->set_mesh_plane();
    plane->set_scale(glm::vec3(10.0f, 10.0f, 10.0f));
    plane->set_position(glm::vec3(0.0f, -1.5f, 0.0f));
    plane->set_materialID(material_2);
    s_meshes.push_back(std::move(plane));

    FaceLightData lit{};
    uint32_t light_0 = Lights::get().createFaceLight(lit);
    auto& light_0_instance = Lights::get().getFaceLight(light_0);
    light_0_instance.setPosition(glm::vec3(0.0f, 1.5f, 0.0f));

    r_camera = std::make_unique<Camera>();
    Celestiq::Input::getInstance().registerKeyCallback(Celestiq::KeyCode::Tab, Celestiq::KeyState::Pressed, [this]{r_camera->resetFirstMouse();});

    initBufferManager();        // 创建缓冲
    uploadSceneToGPU();// 上传数据
}

void Scene::initDescriptor(descriptorPool* pool)
{
    // 创建descriptorSetLayout和descriptorSet
    s_descriptorSetLayout = std::make_unique<descriptorSetLayout>();
    s_descriptorSet = std::make_unique<descriptorSet>();

    VkDescriptorSetLayoutBinding descriptorSetLayoutBinding[7] = {
        // VertexBuffer
        {
            .binding = static_cast<uint32_t>(GlobalSettings::ComputeShaderBinding::VERTEX_BUFFER_SSBO),    //描述符被绑定到0号binding
            .descriptorType = VK_DESCRIPTOR_TYPE_STORAGE_BUFFER,                                           //类型为Storage缓冲区
            .descriptorCount = 1,                                                                          //个数是1个
            .stageFlags = VK_SHADER_STAGE_COMPUTE_BIT                                                      //在计算着色器阶段读取缓冲区
        },
        // IndexBuffer
        {
            .binding = static_cast<uint32_t>(GlobalSettings::ComputeShaderBinding::INDEX_BUFFER_SSBO),     // 1                              
            .descriptorType = VK_DESCRIPTOR_TYPE_STORAGE_BUFFER,    
            .descriptorCount = 1,                                   
            .stageFlags = VK_SHADER_STAGE_COMPUTE_BIT               
        },
        // Material
        {
            .binding = static_cast<uint32_t>(GlobalSettings::ComputeShaderBinding::MATERIAL_BUFFER_SSBO),                                           
            .descriptorType = VK_DESCRIPTOR_TYPE_STORAGE_BUFFER,    
            .descriptorCount = 1,                                   
            .stageFlags = VK_SHADER_STAGE_COMPUTE_BIT               
        },
        // Object
        {
            .binding = static_cast<uint32_t>(GlobalSettings::ComputeShaderBinding::OBJECT_BUFFER_SSBO),                                           
            .descriptorType = VK_DESCRIPTOR_TYPE_STORAGE_BUFFER,    
            .descriptorCount = 1,                                   
            .stageFlags = VK_SHADER_STAGE_COMPUTE_BIT               
        },
        // 摄像机
        {
            .binding = static_cast<uint32_t>(GlobalSettings::ComputeShaderBinding::CAMERA_UBO),                                           
            .descriptorType = VK_DESCRIPTOR_TYPE_UNIFORM_BUFFER,    
            .descriptorCount = 1,                                   
            .stageFlags = VK_SHADER_STAGE_COMPUTE_BIT               
        },
        // Face Light
        {
            .binding = static_cast<uint32_t>(GlobalSettings::ComputeShaderBinding::FACE_LIGHT_SSBO),                                           
            .descriptorType = VK_DESCRIPTOR_TYPE_STORAGE_BUFFER,    
            .descriptorCount = 1,                                   
            .stageFlags = VK_SHADER_STAGE_COMPUTE_BIT               
        },
        // Directional Light
        {
            .binding = static_cast<uint32_t>(GlobalSettings::ComputeShaderBinding::DIRECTIONAL_LIGHT_SSBO),                                           
            .descriptorType = VK_DESCRIPTOR_TYPE_STORAGE_BUFFER,    
            .descriptorCount = 1,                                   
            .stageFlags = VK_SHADER_STAGE_COMPUTE_BIT               
        }
    };

    VkDescriptorSetLayoutCreateInfo descriptorSetLayoutCreateInfo = {
        .bindingCount = 7,
        .pBindings = descriptorSetLayoutBinding
    };

    s_descriptorSetLayout->Create(descriptorSetLayoutCreateInfo);
    pool->AllocateSets(makeSpanFromOne(s_descriptorSet.get()), makeSpanFromOne(s_descriptorSetLayout.get()));
}

void Scene::writeDescriptor()
{
    uint32_t binding_idx = 0;
    s_vertexBufferMgr->writeDescriptor(*s_descriptorSet, binding_idx++);
    s_indexBufferMgr->writeDescriptor(*s_descriptorSet, binding_idx++);
    s_materialBufferMgr->writeDescriptor(*s_descriptorSet, binding_idx++);
    s_objectBufferMgr->writeDescriptor(*s_descriptorSet, binding_idx++);

    // Camera 单独处理
    VkDescriptorBufferInfo bufferInfo_camera = {
        .buffer = r_camera->getCameraUBO(),
        .offset = 0,
        .range = r_camera->getCameraUBOSize()
    };
    s_descriptorSet->Write(makeSpanFromOne(bufferInfo_camera), VK_DESCRIPTOR_TYPE_UNIFORM_BUFFER, binding_idx++);

    s_faceLightBufferMgr->writeDescriptor(*s_descriptorSet, binding_idx++);
    s_directionalLightBufferMgr->writeDescriptor(*s_descriptorSet, binding_idx++);
}

void Scene::update(float deltaTime)
{
    // 更新摄像机
    r_camera->processKey(Celestiq::Input::IsKeyDown(Celestiq::KeyCode::W),
                         Celestiq::Input::IsKeyDown(Celestiq::KeyCode::A),
                         Celestiq::Input::IsKeyDown(Celestiq::KeyCode::S),
                         Celestiq::Input::IsKeyDown(Celestiq::KeyCode::D),
                         deltaTime);
    if(Celestiq::Input::getInstance().GetCursorMode() == Celestiq::CursorMode::Locked)
        r_camera->processMouseMove(Celestiq::Input::GetMousePosition().x,
                                   Celestiq::Input::GetMousePosition().y);
}

void Scene::initBufferManager()
{
    // 构造并收集 Mesh
    s_vertexBufferMgr = std::make_unique<VertexBufferManager>();
    s_vertexBufferMgr->collect(s_meshes);
    s_vertexBufferMgr->create();

    s_indexBufferMgr = std::make_unique<IndexBufferManager>();
    s_indexBufferMgr->collect(s_meshes);
    s_indexBufferMgr->create();

    s_objectBufferMgr = std::make_unique<ObjectBufferManager>();
    s_objectBufferMgr->collect(s_meshes);
    s_objectBufferMgr->create();

    s_materialBufferMgr = std::make_unique<MaterialBufferManager>();
    s_materialBufferMgr->create();

    s_faceLightBufferMgr = std::make_unique<FaceLightBufferManager>();
    s_faceLightBufferMgr->create();

    s_directionalLightBufferMgr = std::make_unique<DirectionalLightBufferManager>();
    s_directionalLightBufferMgr->create();
}

void Scene::uploadSceneToGPU()
{
    s_vertexBufferMgr->upload();
    s_indexBufferMgr->upload();
    s_objectBufferMgr->upload();
    s_materialBufferMgr->upload();
    s_faceLightBufferMgr->upload();
    s_directionalLightBufferMgr->upload();
}


glm::vec3 Scene::hexToVec3(const std::string &hexStr)
{
    std::string hex = hexStr;

    // 移除开头的'#'字符
    if (!hex.empty() && hex[0] == '#') {
        hex.erase(0, 1);
    }

    // 处理三位缩写格式（如"F0F"扩展为"FF00FF"）
    if (hex.length() == 3) {
        std::string expanded;
        for (char c : hex) {
            expanded += std::string(2, c); // 每个字符重复两次
        }
        hex = expanded;
    }

    // 验证长度是否为6位
    if (hex.length() != 6) {
        return glm::vec3(0.0f); // 返回黑色作为默认值
    }

    // 分割RGB分量
    unsigned int r, g, b;
    try {
        r = std::stoul(hex.substr(0, 2), nullptr, 16);
        g = std::stoul(hex.substr(2, 2), nullptr, 16);
        b = std::stoul(hex.substr(4, 2), nullptr, 16);
    } catch (...) {
        return glm::vec3(0.0f); // 解析失败返回黑色
    }

    // 转换为0.0-1.0范围的浮点数
    return glm::vec3(r / 255.0f, g / 255.0f, b / 255.0f);
}