#include "Renderer.h"
#include <GLFW/glfw3.h>
#include "Input.h"

bool renderableImageAttachment::Init(VkExtent2D extent, VkFormat format, VkImageUsageFlags usage, VkImageAspectFlags aspect, descriptorPool* pool)
{
    // Create VkImage
    VkImageCreateInfo imageCreateInfo = {
        .imageType = VK_IMAGE_TYPE_2D,
        .format = format,
        .extent = { extent.width, extent.height, 1 },
        .mipLevels = 1,
        .arrayLayers = 1,
        .samples = VK_SAMPLE_COUNT_1_BIT,
        .tiling = VK_IMAGE_TILING_OPTIMAL,
        .usage = usage | VK_IMAGE_USAGE_SAMPLED_BIT,  // Important!
        .sharingMode = VK_SHARING_MODE_EXCLUSIVE,
        .initialLayout = VK_IMAGE_LAYOUT_UNDEFINED
    };

    imageMem = std::make_unique<imageMemory>(imageCreateInfo, VK_MEMORY_PROPERTY_DEVICE_LOCAL_BIT);

    // Create image view
    VkImageSubresourceRange range = {
        .aspectMask = aspect,
        .baseMipLevel = 0,
        .levelCount = 1,
        .baseArrayLayer = 0,
        .layerCount = 1
    };
    view = std::make_unique<imageView>(imageMem->Image(), VK_IMAGE_VIEW_TYPE_2D, format, range);

    // 配置采样器
    VkSamplerCreateInfo samplerInfo{ VK_STRUCTURE_TYPE_SAMPLER_CREATE_INFO };
    samplerInfo.magFilter    = VK_FILTER_LINEAR;
    samplerInfo.minFilter    = VK_FILTER_LINEAR;
    samplerInfo.mipmapMode   = VK_SAMPLER_MIPMAP_MODE_LINEAR;
    samplerInfo.addressModeU = VK_SAMPLER_ADDRESS_MODE_CLAMP_TO_EDGE;
    samplerInfo.addressModeV = VK_SAMPLER_ADDRESS_MODE_CLAMP_TO_EDGE;
    samplerInfo.addressModeW = VK_SAMPLER_ADDRESS_MODE_CLAMP_TO_EDGE;
    samplerInfo.maxLod       = 1.0f;
    // Create sampler
    samp = std::make_unique<sampler>(samplerInfo);

    // Descriptor layout (binding 0: combined image sampler)
    VkDescriptorSetLayoutBinding binding = {
        .binding = 0,
        .descriptorType = VK_DESCRIPTOR_TYPE_COMBINED_IMAGE_SAMPLER,
        .descriptorCount = 1,
        .stageFlags = VK_SHADER_STAGE_FRAGMENT_BIT
    };
    VkDescriptorSetLayoutCreateInfo layoutInfo = {
        .bindingCount = 1,
        .pBindings = &binding
    };
    layout = std::make_unique<descriptorSetLayout>(layoutInfo);

    set = std::make_unique<descriptorSet>();
    // Allocate descriptor set
    if (pool->AllocateSets(makeSpanFromOne(set.get()), makeSpanFromOne(layout.get()))) return false;

    // Update descriptor
    VkDescriptorImageInfo imgInfo = {
        .sampler = samp->getHandle(),
        .imageView = view->getHandle(),
        .imageLayout = VK_IMAGE_LAYOUT_SHADER_READ_ONLY_OPTIMAL
    };
    set->Write({ &imgInfo, 1 }, VK_DESCRIPTOR_TYPE_COMBINED_IMAGE_SAMPLER);

    return true;
}

bool renderableImageAttachment::Resize(VkExtent2D extent,
                                       VkFormat format,
                                       VkImageUsageFlags usage,
                                       VkImageAspectFlags aspect,
                                       descriptorPool* pool)
{
    // 这里延后一帧销毁旧的 set + imageview 不然ImGui::Image 会依旧持有旧的描述符集，无法释放
    if(set_old) {
        pool->FreeSets(makeSpanFromOne(set_old.get())); // 先释放描述符集，否则图像视图会被引用计数器保护，无法释放
        set_old.reset();
    }
    if(view_old) view_old.reset();
    if(imageMem_old) imageMem_old.reset();

    // 重建 VkImage
    VkImageCreateInfo imageCreateInfo{ VK_STRUCTURE_TYPE_IMAGE_CREATE_INFO };
    imageCreateInfo.imageType     = VK_IMAGE_TYPE_2D;
    imageCreateInfo.format        = format;
    imageCreateInfo.extent        = { extent.width, extent.height, 1 };
    imageCreateInfo.mipLevels     = 1;
    imageCreateInfo.arrayLayers   = 1;
    imageCreateInfo.samples       = VK_SAMPLE_COUNT_1_BIT;
    imageCreateInfo.tiling        = VK_IMAGE_TILING_OPTIMAL;
    imageCreateInfo.usage         = usage | VK_IMAGE_USAGE_SAMPLED_BIT;
    imageCreateInfo.sharingMode   = VK_SHARING_MODE_EXCLUSIVE;
    imageCreateInfo.initialLayout = VK_IMAGE_LAYOUT_UNDEFINED;

    std::unique_ptr<imageMemory> imageMem_new = std::make_unique<imageMemory>(imageCreateInfo,
                                             VK_MEMORY_PROPERTY_DEVICE_LOCAL_BIT);

    // 重建 ImageView
    VkImageSubresourceRange range{
        .aspectMask     = aspect,
        .baseMipLevel   = 0,
        .levelCount     = 1,
        .baseArrayLayer = 0,
        .layerCount     = 1
    };
    std::unique_ptr<imageView> view_new = std::make_unique<imageView>(imageMem_new->Image(),
                                       VK_IMAGE_VIEW_TYPE_2D,
                                       format,
                                       range);


    // 重建描述符集
    std::unique_ptr<descriptorSet> set_new = std::make_unique<descriptorSet>();
    // Allocate descriptor set
    if (pool->AllocateSets(makeSpanFromOne(set_new.get()), makeSpanFromOne(layout.get()))) return false;

    VkDescriptorImageInfo imgInfo{
        .sampler     = samp->getHandle(),
        .imageView   = view_new->getHandle(),
        .imageLayout = VK_IMAGE_LAYOUT_SHADER_READ_ONLY_OPTIMAL
    };
    set_new->Write({ &imgInfo, 1 },
               VK_DESCRIPTOR_TYPE_COMBINED_IMAGE_SAMPLER);

    // 保存到后一帧销毁
    view_old = std::move(view);
    imageMem_old = std::move(imageMem);
    set_old = std::move(set);

    view = std::move(view_new);
    imageMem = std::move(imageMem_new);
    set = std::move(set_new);

    return true;
}


void Renderer::Init()
{
    // 创建描述符池
    std::vector<VkDescriptorPoolSize> poolSizes = {
        { VK_DESCRIPTOR_TYPE_COMBINED_IMAGE_SAMPLER, 100 },
        { VK_DESCRIPTOR_TYPE_UNIFORM_BUFFER, 100 }
    };
    r_descriptorPool = std::make_unique<descriptorPool>(100, poolSizes, VK_DESCRIPTOR_POOL_CREATE_FREE_DESCRIPTOR_SET_BIT);

    r_commandBuffer = std::make_unique<commandBuffer>();
    Celestiq::Application::Get().GetCommandPoolGraphics()->AllocateBuffers(makeSpanFromOne(r_commandBuffer.get()));

    // 栅栏
    r_fence = std::make_unique<fence>();
    // r_semaphore_imageIsAvailable = std::make_unique<semaphore>();
    // r_semaphore_renderingIsOver = std::make_unique<semaphore>();

    // 配置采样器
    VkSamplerCreateInfo samplerInfo{ VK_STRUCTURE_TYPE_SAMPLER_CREATE_INFO };
    samplerInfo.magFilter    = VK_FILTER_LINEAR;
    samplerInfo.minFilter    = VK_FILTER_LINEAR;
    samplerInfo.mipmapMode   = VK_SAMPLER_MIPMAP_MODE_LINEAR;
    samplerInfo.addressModeU = VK_SAMPLER_ADDRESS_MODE_CLAMP_TO_EDGE;
    samplerInfo.addressModeV = VK_SAMPLER_ADDRESS_MODE_CLAMP_TO_EDGE;
    samplerInfo.addressModeW = VK_SAMPLER_ADDRESS_MODE_CLAMP_TO_EDGE;
    samplerInfo.maxLod       = 1.0f;
    // Create sampler
    samp = std::make_unique<sampler>(samplerInfo);

    // G-buffer用 RenderPass-----------------------------------------------------------------------------
    std::array<VkAttachmentDescription, 1> attachments{};

    // Position
    attachments[0].format         = VK_FORMAT_R16G16B16A16_SFLOAT;
    attachments[0].samples        = VK_SAMPLE_COUNT_1_BIT;
    attachments[0].loadOp         = VK_ATTACHMENT_LOAD_OP_CLEAR;
    attachments[0].storeOp        = VK_ATTACHMENT_STORE_OP_STORE;
    attachments[0].stencilLoadOp  = VK_ATTACHMENT_LOAD_OP_DONT_CARE;
    attachments[0].stencilStoreOp = VK_ATTACHMENT_STORE_OP_DONT_CARE;
    attachments[0].initialLayout  = VK_IMAGE_LAYOUT_UNDEFINED;
    attachments[0].finalLayout    = VK_IMAGE_LAYOUT_SHADER_READ_ONLY_OPTIMAL;

    // Normal, Albedo等可选...
    // attachments[1] = attachments[0];
    // attachments[1].format = VK_FORMAT_A2B10G10R10_UNORM_PACK32;

    // Attachment References
    std::array<VkAttachmentReference, 1> colorRefs{};
    for (uint32_t i = 0; i < colorRefs.size(); ++i)
    {
        colorRefs[i].attachment = i;
        colorRefs[i].layout = VK_IMAGE_LAYOUT_COLOR_ATTACHMENT_OPTIMAL;
    }

    // Subpass
    VkSubpassDescription subpass_gbuffer{};
    subpass_gbuffer.pipelineBindPoint = VK_PIPELINE_BIND_POINT_GRAPHICS;
    subpass_gbuffer.colorAttachmentCount = static_cast<uint32_t>(colorRefs.size());
    subpass_gbuffer.pColorAttachments = colorRefs.data();

    // Dependency
    VkSubpassDependency dependency_gbuffer{};
    dependency_gbuffer.srcSubpass = VK_SUBPASS_EXTERNAL;
    dependency_gbuffer.dstSubpass = 0;
    dependency_gbuffer.srcStageMask = VK_PIPELINE_STAGE_BOTTOM_OF_PIPE_BIT;
    dependency_gbuffer.dstStageMask = VK_PIPELINE_STAGE_COLOR_ATTACHMENT_OUTPUT_BIT;
    dependency_gbuffer.srcAccessMask = 0;
    dependency_gbuffer.dstAccessMask = VK_ACCESS_COLOR_ATTACHMENT_WRITE_BIT;

    // Create RenderPass
    VkRenderPassCreateInfo renderPassInfo{ VK_STRUCTURE_TYPE_RENDER_PASS_CREATE_INFO };
    renderPassInfo.attachmentCount = static_cast<uint32_t>(attachments.size());
    renderPassInfo.pAttachments = attachments.data();
    renderPassInfo.subpassCount = 1;
    renderPassInfo.pSubpasses = &subpass_gbuffer;
    renderPassInfo.dependencyCount = 1;
    renderPassInfo.pDependencies = &dependency_gbuffer;

    r_renderPass_gbuffer = std::make_unique<renderPass>(renderPassInfo);

    // 光照阶段用 RenderPass----------------------------------------------------------------------------------------------------
    VkAttachmentDescription colorAttachment{};
    colorAttachment.format         = imageFormat;
    colorAttachment.samples        = VK_SAMPLE_COUNT_1_BIT;
    colorAttachment.loadOp         = VK_ATTACHMENT_LOAD_OP_CLEAR;
    colorAttachment.storeOp        = VK_ATTACHMENT_STORE_OP_STORE;
    colorAttachment.stencilLoadOp  = VK_ATTACHMENT_LOAD_OP_DONT_CARE;
    colorAttachment.stencilStoreOp = VK_ATTACHMENT_STORE_OP_DONT_CARE;
    colorAttachment.initialLayout  = VK_IMAGE_LAYOUT_UNDEFINED;
    // 这里把 finalLayout 改成 SHADER_READ_ONLY，以便后面 ImGui::Image 能直接采样
    colorAttachment.finalLayout    = VK_IMAGE_LAYOUT_SHADER_READ_ONLY_OPTIMAL;

    VkAttachmentReference colorRef{};
    colorRef.attachment = 0;
    colorRef.layout     = VK_IMAGE_LAYOUT_COLOR_ATTACHMENT_OPTIMAL;

    VkSubpassDescription subpass{};
    subpass.pipelineBindPoint    = VK_PIPELINE_BIND_POINT_GRAPHICS;
    subpass.colorAttachmentCount = 1;
    subpass.pColorAttachments    = &colorRef;

    VkSubpassDependency dependency{};
    dependency.srcSubpass    = VK_SUBPASS_EXTERNAL;
    dependency.dstSubpass    = 0;
    dependency.srcStageMask  = VK_PIPELINE_STAGE_COLOR_ATTACHMENT_OUTPUT_BIT;
    dependency.dstStageMask  = VK_PIPELINE_STAGE_FRAGMENT_SHADER_BIT;
    dependency.srcAccessMask = VK_ACCESS_COLOR_ATTACHMENT_WRITE_BIT;
    dependency.dstAccessMask = VK_ACCESS_SHADER_READ_BIT;

    VkRenderPassCreateInfo rpInfo{ VK_STRUCTURE_TYPE_RENDER_PASS_CREATE_INFO };
    rpInfo.attachmentCount = 1;
    rpInfo.pAttachments    = &colorAttachment;
    rpInfo.subpassCount    = 1;
    rpInfo.pSubpasses      = &subpass;
    rpInfo.dependencyCount = 1;
    rpInfo.pDependencies   = &dependency;

    r_renderPass = std::make_unique<renderPass>(rpInfo);

    // G-buffer阶段图像附件及Framebuffer----------------------------------------------------------
    r_gbuffer_position = std::make_unique<renderableImageAttachment>();
    r_gbuffer_position->Init(extent,
                             VK_FORMAT_R16G16B16A16_SFLOAT,
                             VK_IMAGE_USAGE_COLOR_ATTACHMENT_BIT,
                             VK_IMAGE_ASPECT_COLOR_BIT,
                             r_descriptorPool.get());
                        
    std::array<VkImageView, 1> g_buffer_attachments = {
        r_gbuffer_position->GetImageView()
    };

    VkFramebufferCreateInfo framebufferInfo{ VK_STRUCTURE_TYPE_FRAMEBUFFER_CREATE_INFO };
    framebufferInfo.renderPass = r_renderPass_gbuffer->getHandle();
    framebufferInfo.attachmentCount = static_cast<uint32_t>(g_buffer_attachments.size());
    framebufferInfo.pAttachments = g_buffer_attachments.data();
    framebufferInfo.width = extent.width;
    framebufferInfo.height = extent.height;
    framebufferInfo.layers = 1;

    r_framebuffer_gbuffer = std::make_unique<framebuffer>();
    r_framebuffer_gbuffer->Create(framebufferInfo);

    // 光照阶段图像附件及Framebuffer----------------------------------------------------------
    r_onRenderImage = std::make_unique<renderableImageAttachment>();
    if (!r_onRenderImage->Init(extent,
                               imageFormat,
                               VK_IMAGE_USAGE_COLOR_ATTACHMENT_BIT,
                               VK_IMAGE_ASPECT_COLOR_BIT,         
                               r_descriptorPool.get())) {
        throw std::runtime_error("Failed to init renderableImageAttachment");
    }

    VkImageView attachment = r_onRenderImage->GetImageView();
    VkFramebufferCreateInfo fbInfo{ VK_STRUCTURE_TYPE_FRAMEBUFFER_CREATE_INFO };
    fbInfo.renderPass      = r_renderPass->getHandle();
    fbInfo.attachmentCount = 1;
    fbInfo.pAttachments    = &attachment;
    fbInfo.width           = extent.width;
    fbInfo.height          = extent.height;
    fbInfo.layers          = 1;

    r_framebuffers = std::make_unique<framebuffer>();
    r_framebuffers->Create(fbInfo);

    // 创建g-buffer管线 -------------------------------------------------------------------
    // 先创建摄像机（描述符集）
    r_camera = std::make_unique<Camera>();
    Celestiq::Input::getInstance().registerKeyCallback(Celestiq::KeyCode::Tab, Celestiq::KeyState::Pressed, [this]{r_camera->resetFirstMouse();});
    r_descriptorSetLayout_camera = std::make_unique<descriptorSetLayout>();
    r_descriptorSet_camera = std::make_unique<descriptorSet>();
    VkDescriptorSetLayoutBinding descriptorSetLayoutBinding_camera = {
        .binding = 0,                                       //描述符被绑定到0号binding
        .descriptorType = VK_DESCRIPTOR_TYPE_UNIFORM_BUFFER,//类型为uniform缓冲区
        .descriptorCount = 1,                               //个数是1个
        .stageFlags = VK_SHADER_STAGE_VERTEX_BIT            //在顶点着色器阶段读取uniform缓冲区
    };
    VkDescriptorSetLayoutCreateInfo descriptorSetLayoutCreateInfo_camera = {
        .bindingCount = 1,
        .pBindings = &descriptorSetLayoutBinding_camera
    };
    r_descriptorSetLayout_camera->Create(descriptorSetLayoutCreateInfo_camera);
    r_descriptorPool->AllocateSets(makeSpanFromOne(r_descriptorSet_camera.get()), makeSpanFromOne(r_descriptorSetLayout_camera.get()));
    VkDescriptorBufferInfo bufferInfo = {
        .buffer = r_camera->getCameraUBO(),
        .offset = 0,
        .range = r_camera->getCameraUBOSize()//或VK_WHOLE_SIZE
    };
    r_descriptorSet_camera->Write(makeSpanFromOne(bufferInfo), VK_DESCRIPTOR_TYPE_UNIFORM_BUFFER);

    VkPipelineLayoutCreateInfo pipelineLayoutCreateInfo_gbuffer = {
        .setLayoutCount = 1,
        .pSetLayouts = r_descriptorSetLayout_camera->Address()
    };
    r_pipelineLayout_gbuffer = std::make_unique<pipelineLayout>(pipelineLayoutCreateInfo_gbuffer);

    // 创建着色器
    r_shaders["vert_gbuffer"] = std::make_unique<shaderModule>("res/shader_spv/Deferred_G.vert.spv");
    r_shaders["frag_gbuffer"] = std::make_unique<shaderModule>("res/shader_spv/Deferred_G.frag.spv");
    static VkPipelineShaderStageCreateInfo shaderStageCreateInfos_gbuffer[2] = {
        r_shaders["vert_gbuffer"]->StageCreateInfo(VK_SHADER_STAGE_VERTEX_BIT),
        r_shaders["frag_gbuffer"]->StageCreateInfo(VK_SHADER_STAGE_FRAGMENT_BIT)
    };

    graphicsPipelineCreateInfoPack pipelineCiPack_gbuffer;
    pipelineCiPack_gbuffer.createInfo.layout = r_pipelineLayout_gbuffer->getHandle();
    pipelineCiPack_gbuffer.createInfo.renderPass = r_renderPass_gbuffer->getHandle();
    pipelineCiPack_gbuffer.inputAssemblyStateCi.topology = VK_PRIMITIVE_TOPOLOGY_TRIANGLE_LIST;
    pipelineCiPack_gbuffer.rasterizationStateCi.lineWidth = 1.0f;
    pipelineCiPack_gbuffer.multisampleStateCi.rasterizationSamples = VK_SAMPLE_COUNT_1_BIT;
    pipelineCiPack_gbuffer.colorBlendAttachmentStates.push_back({ .blendEnable = VK_FALSE, .colorWriteMask = 0b1111 });
    pipelineCiPack_gbuffer.dynamicStates.push_back(VK_DYNAMIC_STATE_VIEWPORT);
    pipelineCiPack_gbuffer.dynamicStates.push_back(VK_DYNAMIC_STATE_SCISSOR);
    pipelineCiPack_gbuffer.UpdateAllArrays();
    pipelineCiPack_gbuffer.createInfo.stageCount = 2;
    pipelineCiPack_gbuffer.createInfo.pStages = shaderStageCreateInfos_gbuffer;
    Mesh::bind_pipeline(pipelineCiPack_gbuffer);
    r_pipeline_gbuffer = std::make_unique<pipeline>(pipelineCiPack_gbuffer);

    // 创建光照管线 ------------------------------------------------------------------- 
    // 为g-buffer图像创建描述符集
    r_descriptorSetLayout_lighting = std::make_unique<descriptorSetLayout>();
    r_descriptorSet_lighting = std::make_unique<descriptorSet>();
    std::array<VkDescriptorSetLayoutBinding, 1> bindings;
    for (int i = 0; i < 1; ++i) {
        bindings[i] = {
            .binding = static_cast<uint32_t>(i),
            .descriptorType = VK_DESCRIPTOR_TYPE_COMBINED_IMAGE_SAMPLER,
            .descriptorCount = 1,
            .stageFlags = VK_SHADER_STAGE_FRAGMENT_BIT
        };
    }
    VkDescriptorSetLayoutCreateInfo descriptorSetLayoutCreateInfo_lighting = {
        .bindingCount = bindings.size(),
        .pBindings = bindings.data()
    };
    r_descriptorSetLayout_lighting->Create(descriptorSetLayoutCreateInfo_lighting);
    r_descriptorPool->AllocateSets(makeSpanFromOne(r_descriptorSet_lighting.get()), makeSpanFromOne(r_descriptorSetLayout_lighting.get()));
    // VkDescriptorImageInfo infos[] = {
    //     { .sampler = samp->getHandle(), .imageView = r_gbuffer_position->GetImageView(), .imageLayout = VK_IMAGE_LAYOUT_SHADER_READ_ONLY_OPTIMAL }
    // };
    // r_descriptorSet_lighting->Write(makeSpanFromOne(infos), VK_DESCRIPTOR_TYPE_COMBINED_IMAGE_SAMPLER);

    
    VkPipelineLayoutCreateInfo pipelineLayoutInfo = {
        .sType = VK_STRUCTURE_TYPE_PIPELINE_LAYOUT_CREATE_INFO,
        .setLayoutCount = 1,
        .pSetLayouts = r_descriptorSetLayout_lighting->Address()
    };

    r_pipelineLayout = std::make_unique<pipelineLayout>(pipelineLayoutInfo);

    // 创建着色器
    r_shaders["vert"] = std::make_unique<shaderModule>("res/shader_spv/Deferred_L.vert.spv");
    r_shaders["frag"] = std::make_unique<shaderModule>("res/shader_spv/Deferred_L.frag.spv");
    static VkPipelineShaderStageCreateInfo shaderStageCreateInfos[2] = {
        r_shaders["vert"]->StageCreateInfo(VK_SHADER_STAGE_VERTEX_BIT),
        r_shaders["frag"]->StageCreateInfo(VK_SHADER_STAGE_FRAGMENT_BIT)
    };

    graphicsPipelineCreateInfoPack pipelineCiPack;
    pipelineCiPack.createInfo.layout = r_pipelineLayout->getHandle();
    pipelineCiPack.createInfo.renderPass = r_renderPass->getHandle();
    pipelineCiPack.inputAssemblyStateCi.topology = VK_PRIMITIVE_TOPOLOGY_TRIANGLE_LIST;
    pipelineCiPack.rasterizationStateCi.lineWidth = 1.0f;
    pipelineCiPack.multisampleStateCi.rasterizationSamples = VK_SAMPLE_COUNT_1_BIT;
    pipelineCiPack.colorBlendAttachmentStates.push_back({ .colorWriteMask = 0b1111 });
    pipelineCiPack.dynamicStates.push_back(VK_DYNAMIC_STATE_VIEWPORT);
    pipelineCiPack.dynamicStates.push_back(VK_DYNAMIC_STATE_SCISSOR);
    pipelineCiPack.UpdateAllArrays();
    pipelineCiPack.createInfo.stageCount = 2;
    pipelineCiPack.createInfo.pStages = shaderStageCreateInfos;
    Mesh::bind_pipeline(pipelineCiPack);
    r_pipeline = std::make_unique<pipeline>(pipelineCiPack);
// -----------------------------------------------------------------------------------------

    r_mesh_cube = std::make_unique<Mesh>();
    r_mesh_cube->set_mesh_cube();
    r_mesh_screen = std::make_unique<Mesh>();
    r_mesh_screen->set_mesh_screen();
}

void Renderer::drawFrame(VkExtent2D newExtent)
{
    // 计算帧时间
    currentTime = glfwGetTime();
    deltaTime = currentTime - lastTime;
    lastTime = currentTime;

    // 更新摄像机
    r_camera->processKey(Celestiq::Input::IsKeyDown(Celestiq::KeyCode::W),
                         Celestiq::Input::IsKeyDown(Celestiq::KeyCode::A),
                         Celestiq::Input::IsKeyDown(Celestiq::KeyCode::S),
                         Celestiq::Input::IsKeyDown(Celestiq::KeyCode::D),
                         deltaTime);
    if(Celestiq::Input::getInstance().GetCursorMode() == Celestiq::CursorMode::Locked)
        r_camera->processMouseMove(Celestiq::Input::GetMousePosition().x,
                                   Celestiq::Input::GetMousePosition().y);
    // 处理输入
    Celestiq::Input::getInstance().update();

    resizeImageFramebuffers(newExtent);

    r_commandBuffer->Begin(VK_COMMAND_BUFFER_USAGE_ONE_TIME_SUBMIT_BIT);
    // GBuffer Pass-----------------------------------------------------------------
    VkClearValue clearColor_gbuffer[1] = {};
    for (int i = 0; i < 1; ++i) {
        clearColor_gbuffer[i].color = { {0.0f, 0.0f, 0.0f, 1.0f} };
    }
    r_renderPass_gbuffer->CmdBegin(r_commandBuffer->getHandle(),
                                   r_framebuffer_gbuffer->getHandle(),
                                   { {}, newExtent },
                                   clearColor_gbuffer);
    // 设置视口 & 裁剪
    resizePipeline(newExtent, r_commandBuffer->getHandle());
    // 绑定 GBuffer 管线
    vkCmdBindPipeline(r_commandBuffer->getHandle(),
                    VK_PIPELINE_BIND_POINT_GRAPHICS,
                    r_pipeline_gbuffer->getHandle());
    vkCmdBindDescriptorSets(r_commandBuffer->getHandle(), VK_PIPELINE_BIND_POINT_GRAPHICS,
        r_pipelineLayout_gbuffer->getHandle(), 0, 1, r_descriptorSet_camera->Address(), 0, nullptr);
    // 绘制 Mesh
    r_mesh_cube->draw(r_commandBuffer->getHandle());
    r_renderPass_gbuffer->CmdEnd(r_commandBuffer->getHandle());

    // 管线屏障，等待 GBuffer 阶段结束
    // transitionAttachmentsToShaderRead(
    //     r_commandBuffer->getHandle(),
    //     {
    //         r_gbuffer_position.get()
    //     }
    // );
    // 光照pass------------------------------------------------------------------------
    VkClearValue clearColor = { .color = { 0.0f, 0.0f, 0.0f, 1.f } };
    r_renderPass->CmdBegin(r_commandBuffer->getHandle(), r_framebuffers->getHandle(), { {}, newExtent }, makeSpanFromOne(clearColor));
    resizePipeline(newExtent, r_commandBuffer->getHandle());
    /*渲染命令*/
    vkCmdBindDescriptorSets(r_commandBuffer->getHandle(), VK_PIPELINE_BIND_POINT_GRAPHICS,
        r_pipelineLayout->getHandle(), 0, 1, r_descriptorSet_lighting->Address(), 0, nullptr);
    vkCmdBindPipeline(r_commandBuffer->getHandle(), VK_PIPELINE_BIND_POINT_GRAPHICS, r_pipeline->getHandle());
    r_mesh_screen->draw(r_commandBuffer->getHandle());
    /*渲染命令结束*/
    r_renderPass->CmdEnd(r_commandBuffer->getHandle());

    r_commandBuffer->End();
    SubmitCommandBuffer_Graphics(r_commandBuffer->getHandle(), r_fence->getHandle());

    extent = newExtent;
    r_fence->WaitAndReset();
    ImGui::Image((ImTextureID)r_onRenderImage->GetDescriptorSet(), ImVec2((float)extent.width, (float)extent.height), ImVec2(0, 1), ImVec2(1, 0));
}

void Renderer::resizeImageFramebuffers(VkExtent2D newExtent)
{
    // 零尺寸保护
    if (newExtent.width == 0 || newExtent.height == 0){
        std::cout << "Renderer::resizeImageFramebuffers: zero size" << std::endl;
        return;
    }
    // 无变化直接返回
    if (newExtent.width == extent.width && newExtent.height == extent.height)
        return;

    r_onRenderImage->Resize(newExtent,
                            imageFormat,
                            VK_IMAGE_USAGE_COLOR_ATTACHMENT_BIT,
                            VK_IMAGE_ASPECT_COLOR_BIT,
                            r_descriptorPool.get());

    r_gbuffer_position->Resize(newExtent,
                                  VK_FORMAT_R16G16B16A16_SFLOAT,
                                  VK_IMAGE_USAGE_COLOR_ATTACHMENT_BIT,
                                  VK_IMAGE_ASPECT_COLOR_BIT,
                                  r_descriptorPool.get());

    VkDescriptorImageInfo infos[] = {
        { .sampler = samp->getHandle(), .imageView = r_gbuffer_position->GetImageView(), .imageLayout = VK_IMAGE_LAYOUT_SHADER_READ_ONLY_OPTIMAL }
    };
    r_descriptorSet_lighting->Write(makeSpanFromOne(infos), VK_DESCRIPTOR_TYPE_COMBINED_IMAGE_SAMPLER);
    
    // 销毁旧的 Framebuffer
    r_framebuffers.reset();
    // 用新 Attachment 的 ImageView 创建 Framebuffer
    VkImageView attachment = r_onRenderImage->GetImageView();
    VkFramebufferCreateInfo fbInfo{ VK_STRUCTURE_TYPE_FRAMEBUFFER_CREATE_INFO };
    fbInfo.renderPass      = r_renderPass->getHandle();
    fbInfo.attachmentCount = 1;
    fbInfo.pAttachments    = &attachment;
    fbInfo.width           = newExtent.width;
    fbInfo.height          = newExtent.height;
    fbInfo.layers          = 1;
    r_framebuffers = std::make_unique<framebuffer>();
    r_framebuffers->Create(fbInfo);

    // 销毁旧的 GBuffer Framebuffer
    r_framebuffer_gbuffer.reset();
    std::array<VkImageView, 1> g_buffer_attachments = {
        r_gbuffer_position->GetImageView()
    };
    VkFramebufferCreateInfo framebufferInfo{ VK_STRUCTURE_TYPE_FRAMEBUFFER_CREATE_INFO };
    framebufferInfo.renderPass = r_renderPass_gbuffer->getHandle();
    framebufferInfo.attachmentCount = static_cast<uint32_t>(g_buffer_attachments.size());
    framebufferInfo.pAttachments = g_buffer_attachments.data();
    framebufferInfo.width = newExtent.width;
    framebufferInfo.height = newExtent.height;
    framebufferInfo.layers = 1;
    r_framebuffer_gbuffer = std::make_unique<framebuffer>();
    r_framebuffer_gbuffer->Create(framebufferInfo);
}

void Renderer::resizePipeline(VkExtent2D newExtent, VkCommandBuffer commandBuffer)
{
    // 零尺寸保护
    if (newExtent.width == 0 || newExtent.height == 0){
        std::cout << "Renderer::resizePipeline: zero size" << std::endl;
        return;
    }
    // 每帧都需要设置 viewport 和 scissor，不管有没有变化
    VkViewport viewport{};
    viewport.x = 0.0f;
    viewport.y = 0.0f;
    viewport.width  = static_cast<float>(newExtent.width);
    viewport.height = static_cast<float>(newExtent.height);
    viewport.minDepth = 0.0f;
    viewport.maxDepth = 1.0f;
    vkCmdSetViewport(commandBuffer, 0, 1, &viewport);

    VkRect2D scissor{};
    scissor.offset = {0, 0};
    scissor.extent = newExtent;
    vkCmdSetScissor(commandBuffer, 0, 1, &scissor);
}

VkResult Renderer::SubmitCommandBuffer_Graphics(VkSubmitInfo &submitInfo, VkFence fence)
{
    submitInfo.sType = VK_STRUCTURE_TYPE_SUBMIT_INFO;
    VkResult result = vkQueueSubmit(Celestiq::Application::GetQueue(), 1, &submitInfo, fence);
    if (result)
        std::cout << std::format("[ graphicsBase ] ERROR\nFailed to submit the command buffer!\nError code: {}\n", int32_t(result));
    return result;
}

VkResult Renderer::SubmitCommandBuffer_Graphics(VkCommandBuffer commandBuffer, VkSemaphore semaphore_imageIsAvailable, VkSemaphore semaphore_renderingIsOver, VkFence fence, VkPipelineStageFlags waitDstStage_imageIsAvailable)
{
    VkSubmitInfo submitInfo = {
        .commandBufferCount = 1,
        .pCommandBuffers = &commandBuffer
    };
    if (semaphore_imageIsAvailable)
        submitInfo.waitSemaphoreCount = 1,
        submitInfo.pWaitSemaphores = &semaphore_imageIsAvailable,
        submitInfo.pWaitDstStageMask = &waitDstStage_imageIsAvailable;
    if (semaphore_renderingIsOver)
        submitInfo.signalSemaphoreCount = 1,
        submitInfo.pSignalSemaphores = &semaphore_renderingIsOver;
    return SubmitCommandBuffer_Graphics(submitInfo, fence);
}

VkResult Renderer::SubmitCommandBuffer_Graphics(VkCommandBuffer commandBuffer, VkFence fence)
{
    VkSubmitInfo submitInfo = {
        .commandBufferCount = 1,
        .pCommandBuffers = &commandBuffer
    };
    return SubmitCommandBuffer_Graphics(submitInfo, fence);
}

void Renderer::transitionAttachmentsToShaderRead(VkCommandBuffer cmd,
                                                 const std::vector<renderableImageAttachment*>& attachments)
{
    std::vector<VkImageMemoryBarrier> barriers;
    barriers.reserve(attachments.size());

    for (const auto* attachment : attachments)
    {
        VkImageMemoryBarrier barrier{ VK_STRUCTURE_TYPE_IMAGE_MEMORY_BARRIER };
        barrier.srcAccessMask = VK_ACCESS_COLOR_ATTACHMENT_WRITE_BIT;
        barrier.dstAccessMask = VK_ACCESS_SHADER_READ_BIT;
        barrier.oldLayout     = VK_IMAGE_LAYOUT_COLOR_ATTACHMENT_OPTIMAL;
        barrier.newLayout     = VK_IMAGE_LAYOUT_SHADER_READ_ONLY_OPTIMAL;
        barrier.srcQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
        barrier.dstQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
        barrier.image = attachment->GetImage();
        barrier.subresourceRange.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
        barrier.subresourceRange.baseMipLevel = 0;
        barrier.subresourceRange.levelCount = 1;
        barrier.subresourceRange.baseArrayLayer = 0;
        barrier.subresourceRange.layerCount = 1;
        barriers.push_back(barrier);
    }

    vkCmdPipelineBarrier(cmd,
                         VK_PIPELINE_STAGE_COLOR_ATTACHMENT_OUTPUT_BIT,
                         VK_PIPELINE_STAGE_FRAGMENT_SHADER_BIT,
                         0,
                         0, nullptr,
                         0, nullptr,
                         static_cast<uint32_t>(barriers.size()), barriers.data());
}
